// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_NOTIFICATION_ENDPOINT_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_NOTIFICATION_ENDPOINT_H

#include "common.h"
#include "INotificationEndpointCallbacks.h"
#include "ProtocolEndpointBase.h"
#include "util/shared_ptr.h"

/**
 * This endpoint is used to set up a special notification channel between the phone and
 * the car that allows for the exchange of freeform text notifications. A typical use case
 * for this channel would be to send the phone strings that would normally be rendered as
 * an overlay. The channel is bi-directional, the phone can also send free form notifications
 * to the car. A typical initialization sequence is shown below.
 * <br>
 * <pre>
 *     galReceiver->init();
 *     ... Initialization code ...
 *     NotificationEndpoint* endpoint =
 *             new NotificationEndpoint(serviceId, galReceiver->messageRouter());
 *     endpoint->registerCallbacks(callbacks); // Subclassed from INotificationEndpointCallbacks
 *     galReceiver->registerService(endpoint);
 *     galReceiver->start();
 */
class NotificationEndpoint : public ProtocolEndpointBase {
public:
    NotificationEndpoint(uint8_t id, MessageRouter* router) :
            ProtocolEndpointBase(id, router, false),
            mSubscribed(false) { }
    void addDiscoveryInfo(ServiceDiscoveryResponse* sdr);
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void registerCallbacks(const shared_ptr<INotificationEndpointCallbacks>& callbacks) {
        mCallbacks = callbacks;
    }

    /**
     * Call this method to send a notification to the other end.
     * @param text The text of the notification to display.
     * @param hasId If true, the id parameter contains valid data.
     * @param id An identifier for this notification. Supply this argument only if you wish
     *        to receive an acknowledgement that this notification has been handled.
     * @param hasIcon If true, the icon argument contains valid data.
     * @param icon A pointer to the icon data (png format). The icon can be 96x96 pixels at maximum.
     * @param len The size of the icon in bytes.
     * @return STATUS_SUCCESS on success, an error code otherwise. If this method is called
     *         before the phone sends the subscribe message, the call will fail.
     */
    int sendNotification(const std::string& text, bool hasId,
            const std::string& id, bool hasIcon, uint8_t* icon, size_t len);
    /**
     * Call this method to acknowledge a notification. Calling this method informs the other
     * side that the notification has been communicated to the user (displayed, read aloud etc.)
     * @param id The identifier of the notification.
     * @return STATUS_SUCCESS on success, an error code otherwise. If this method is called
     *         before the phone sends the subscribe message, the call will fail.
     */
    int ackNotification(const std::string& id);

private:
    shared_ptr<INotificationEndpointCallbacks> mCallbacks;
    bool mSubscribed;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_NOTIFICATION_ENDPOINT_H
